/*******************************************************************************
 *
 *  UserExperience.c - Main Application
 *
 *  Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/

#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "msp430.h"
#include "HAL_PMM.h"
#include "HAL_UCS.h"
#include "HAL_TLV.h"
#include "hal_board.h"
#include "hal_buttons.h"
#include "hal_rfspi.h"
//
#include "dispatcher.h"
#include "board_spi.h"

#include "application_version.h"
#include "host_driver_version.h"

/* EE API Includes */
#include "wlan.h" 
#include "evnt_handler.h"    
#include "nvmem.h"
#include "socket.h"
#include "cc3000_common.h"
#include "netapp.h"
#include "spi.h"

#define PALTFORM_VERSION						(1)

#define UART_COMMAND_CC3000_SIMPLE_CONFIG_START	 (0x31)

#define UART_COMMAND_CC3000_CONNECT			 	 (0x32)

#define UART_COMMAND_SOCKET_OPEN			 	 (0x33)

#define UART_COMMAND_SEND_DATA				  	 (0x34)

#define UART_COMMAND_RCV_DATA				  	 (0x35)

#define UART_COMMAND_BSD_BIND					 (0x36)

#define UART_COMMAND_SOCKET_CLOSE			 	 (0x37)

#define UART_COMMAND_IP_CONFIG			 	 	 (0x38)

#define UART_COMMAND_CC3000_DISCONNECT		 	 (0x39)

#define UART_COMMAND_CC3000_DEL_POLICY		 	 (0x61)

#define CC3000_RX_BUFFER_SIZE					  (128)


static void SmartConfigLedOn(uint8_t ulTrueFalse);
static void StartUnsolicitedEventTimer(void);
uint32_t ulSocket, ulSmartConfigFinished;
volatile uint8_t  ulCC3000Connected;

//
// Reception from the air, buffer - the max data length  + headers
//
uint8_t pucCC3000_Rx_Buffer[CC3000_RX_BUFFER_SIZE + 100];
//
// Simple Config Prefix
//
char aucCC3000_prefix[] = {'T', 'T', 'T'};
//
// Indications that UART command has finished etc
//
uint8_t pucUARTCommandDoneString[] = {'\f', '\r', 'D', 'O', 'N', 'E', '\f', '\r'};
uint8_t pucUARTCommandSimpleConfigDoneString[] = {'\f', '\r','S', 'i', 'm', 'p', 'l', 'e', ' ', 'c', 'o', 'n', 'f', 'i', 'g', ' ',  'D', 'O', 'N', 'E', '\f', '\r'};
uint8_t pucUARTExampleAppString[] = {'\f', '\r','E', 'x', 'a', 'm', 'p', 'l', 'e', ' ', 'A', 'p', 'p', ':', 'd', 'r', 'i', 'v', 'e', 'r', ' ', 'v', 'e', 'r', 's', 'i', 'o', 'n', ' ' };
uint8_t pucUARTNoDataString[] = {'\f', '\r', 'N', 'o', ' ', 'd', 'a', 't', 'a', ' ', 'r', 'e', 'c','e', 'i', 'v', 'e', 'd', '\f', '\r'};
uint8_t pucUARTIllegalCommandString[] = {'\f', '\r', 'I', 'l', 'l', 'e', 'g', 'a', 'l', ' ', 'c', 'o','m', 'm', 'a', 'n', 'd', '\f', '\r'};

void CC3000_UsynchCallback(long lEventType, char * data, unsigned char length);
uint8_t *sendDriverPatch(unsigned long *Length);
uint8_t *sendBootLoaderPatch(unsigned long *Length);
uint8_t *sendWLFWPatch(unsigned long *Length);
static void DemoInitSpi(void);

long ReadWlanInterruptPin(void);
void WlanInterruptEnable(void);
void WlanInterruptDisable(void);


/**
  * @brief  Enables the radio
  * @param  True or False
  * @retval None
  */
void WriteWlanPin(uint8_t trueFalse)
{	
	if(trueFalse)
	{
    	P1OUT |= WLAN_EN_PIN;                 // RF_EN_PIN high
	}
	else
	{
    	P1OUT &= ~WLAN_EN_PIN;                // RF_EN_PIN low
	}
	/* This delay is required otherwise the Radio locks up occasionally. */
#if defined(__TI_COMPILER_VERSION__)
	SysCtlDelay(6000000);
#else
	__delay_cycles(6000000);
#endif
}

/**
  * @brief  Enables Interrupt Detection on IRQ input pin
  * @param  None
  * @retval None
  */
void WlanInterruptEnable(void)
{
	halButtonsInterruptEnable(BUTTON_IRQ);
}


/**
  * @brief  Disables Interrupt Detection on IRQ input pin
  * @param  None
  * @retval None
  */
void WlanInterruptDisable(void)
{
	halButtonsInterruptDisable(BUTTON_IRQ);
}

/**
  * @brief  Reads the status of the IRQ pin from CC3000
  * @param  None
  * @retval TRUE or FALSE
  */
long ReadWlanInterruptPin(void)
{
	/* Return the status of IRQ, Port2.bit3 */
    return    (P1IN & WLAN_IRQ_PIN);
}
/**
  * @brief  This function returns a pointer to the driver patch.
  * @param  The length of the patch.
  * @retval None
  */
uint8_t *sendDriverPatch(unsigned long *Length)
{
	*Length = 0;
	return NULL;
}


/**
  * @brief  This function returns a pointer to the bootloader patch.
  * @param  The length of the patch.
  * @retval None
  */
uint8_t *sendBootLoaderPatch(unsigned long *Length)
{
	*Length = 0;
	return NULL;
}


/**
  * @brief  This function returns a pointer to the firmware patch.
  * @param  The length of the patch.
  * @retval None
  */
uint8_t *sendWLFWPatch(unsigned long *Length)
{
	*Length = 0;
	return NULL;
}

/**
  * @brief  This function triggers configuration process of CC3000
  * @param  None
  * @retval None
  */
void StartSmartConfig(void)
{
	ulSmartConfigFinished = 0;
	ulCC3000Connected = 0;

	//
	// Reset all the previous configuration
	//
	wlan_ioctl_set_connection_policy(0,0,0);	
	
	//
	// Trigger the Smart Config process
	//
	SmartConfigLedOn(TRUE);
	wlan_first_time_config_set_prefix(aucCC3000_prefix);
	SmartConfigLedOn(FALSE);	 
	//
	// Start the Smart Config process
	//
	wlan_first_time_config_start();
        
	SmartConfigLedOn(TRUE);	
	//
	// Wait for simple config finished
	// 
	while (ulSmartConfigFinished == 0)
	{
		SmartConfigLedOn(FALSE);
#if defined(__TI_COMPILER_VERSION__)
	SysCtlDelay(6000000);
#else
	__delay_cycles(6000000);
#endif
		SmartConfigLedOn(TRUE);
#if defined(__TI_COMPILER_VERSION__)
	SysCtlDelay(6000000);
#else
	__delay_cycles(6000000);
#endif
		//SysCtlDelay(16500);
	}
	SmartConfigLedOn(TRUE);
	//
	// Configure to connect automatically to the AP retrieved in the 
	// simple config process
	//
	wlan_ioctl_set_connection_policy(0, 0, 1);
	//
	// reset the CC3000
	// 
	wlan_stop();

	DispatcherUartSendPacket((uint8_t *)pucUARTCommandSimpleConfigDoneString, sizeof(pucUARTCommandSimpleConfigDoneString));
	//Buttons_init(BUTTON_ALL);
    //Buttons_interruptEnable(BUTTON_ALL);
	wlan_start(0);
	//
	// Mask out all non-required events
	//
	wlan_set_event_mask(HCI_EVNT_WLAN_KEEPALIVE|HCI_EVNT_WLAN_UNSOL_INIT|HCI_EVNT_WLAN_UNSOL_DHCP|HCI_EVNT_WLAN_ASYNC_PING_REPORT);
}


/**
  * @brief  converts and decimal to char.
  * @param  None
  * @retval None
  */
uint8_t atoc(uint8_t data)
{
	uint8_t ucRes;

	if ((data >= 0x30) && (data <= 0x39))
	{
		ucRes = data - 0x30;
	}
	else
	{
		if (data == 'a')
		{
			ucRes = 0x0a;;
		}
		else if (data == 'b')
		{
			ucRes = 0x0b;
		}
		else if (data == 'c')
		{
			ucRes = 0x0c;
		}
		else if (data == 'd')
		{
			ucRes = 0x0d;
		}
		else if (data == 'e')
		{
			ucRes = 0x0e;
		}
		else if (data == 'f')
		{
			ucRes = 0x0f;
		}
	}
	return ucRes;
}

/**
  * @brief  converts 2 nibbles into a 16-bit number
  * @param  char1
  * @param  char2
  * @retval number
  */
uint16_t atoshort(uint8_t b1, uint8_t b2)
{
	uint16_t usRes;
	usRes = (atoc(b1)) * 16 | atoc(b2);
	return usRes;
}

/**
  * @brief  Converts 2 bytes into an ascii character
  * @param  None
  * @retval None
  */
uint8_t ascii_to_char(uint8_t b1, uint8_t b2)
{
	unsigned char ucRes;

	ucRes = (atoc(b1)) << 4 | (atoc(b2));

	return ucRes;
}

/**
  * @brief  Processes the incoming USART commands.
  * @param  Pointer to buffer holding incoming data
  * @retval None
  */
void DemoHandleUartCommand(uint8_t *usBuffer)
{
	uint8_t *pcSsid, *pcData, *pcSockAddrAscii;
	uint32_t ulSsidLen, ulDataLength;
	volatile uint16_t iReturnValue;
	sockaddr tSocketAddr;
	socklen_t tRxPacketLength;
	uint8_t pucIP_Addr[4];
	uint8_t pucIP_DefaultGWAddr[4];
	uint8_t pucSubnetMask[4];
	uint8_t pucDNS[4];

    switch(usBuffer[1])
    {
    	//
    	// Start a smart configuration process
    	//
    	case UART_COMMAND_CC3000_SIMPLE_CONFIG_START:
			StartSmartConfig();
			break;
		
    	//
    	// Start a WLAN Connect process
    	//
		case UART_COMMAND_CC3000_CONNECT:
        {
            ulSsidLen = atoc(usBuffer[2]);
			pcSsid = &usBuffer[3];
			
            wlan_connect(0, (char *)pcSsid, ulSsidLen, 
            	NULL, NULL, 0);
        } 
		break;
    	//
    	// Handle open socket command
    	//
		case UART_COMMAND_SOCKET_OPEN:
		    ulSocket = socket(2,2,17);
			break;
    	//
    	// Handle close socket command
    	//
		case UART_COMMAND_SOCKET_CLOSE:
			closesocket(ulSocket);
			ulSocket = 0xFFFFFFFF;
			break;	
    	//
    	// Handle receive data command
    	//
		case UART_COMMAND_RCV_DATA:
			iReturnValue = recvfrom(ulSocket, pucCC3000_Rx_Buffer, CC3000_RX_BUFFER_SIZE, 0, &tSocketAddr, &tRxPacketLength);
			if (iReturnValue <= 0)
			{
				//
				// No data receibed by device
				//
				DispatcherUartSendPacket(pucUARTNoDataString, sizeof(pucUARTNoDataString));

			}
			else
			{
				//
				// Send data to UART...
				//
				DispatcherUartSendPacket(pucCC3000_Rx_Buffer, CC3000_RX_BUFFER_SIZE);
			}
			break;

		
    	//
    	// Handle send data command
    	//
		case UART_COMMAND_SEND_DATA:
			pcData = &usBuffer[4];
			ulDataLength = atoshort(usBuffer[2], usBuffer[3]);
			pcSockAddrAscii = (pcData + ulDataLength);
			tSocketAddr.sa_family = atoshort(pcSockAddrAscii[0], pcSockAddrAscii[1]);

			tSocketAddr.sa_data[0] = ascii_to_char(pcSockAddrAscii[2], pcSockAddrAscii[3]);
			tSocketAddr.sa_data[1] = ascii_to_char(pcSockAddrAscii[4], pcSockAddrAscii[5]);
			
			tSocketAddr.sa_data[2] = ascii_to_char(pcSockAddrAscii[6], pcSockAddrAscii[7]);
			tSocketAddr.sa_data[3] = ascii_to_char(pcSockAddrAscii[8], pcSockAddrAscii[9]);
			tSocketAddr.sa_data[4] = ascii_to_char(pcSockAddrAscii[10], pcSockAddrAscii[11]);
			tSocketAddr.sa_data[5] = ascii_to_char(pcSockAddrAscii[12], pcSockAddrAscii[13]);
			
			sendto(ulSocket, pcData, ulDataLength, 0, &tSocketAddr, sizeof(sockaddr));
    		break;

		
    	//
    	// Handle bind command
    	//
		case UART_COMMAND_BSD_BIND:
			tSocketAddr.sa_family = 2;

			tSocketAddr.sa_data[0] = ascii_to_char(usBuffer[2], usBuffer[3]);
			tSocketAddr.sa_data[1] = ascii_to_char(usBuffer[4], usBuffer[5]);
			memset (&tSocketAddr.sa_data[2], 0, 4);
			
			bind(ulSocket, &tSocketAddr, sizeof(sockaddr));
			
			break;

		
    	//
    	// Handle IP configuration command
    	//
		case UART_COMMAND_IP_CONFIG:
			
			//
			// Network mask is assumed to be 255.255.255.0
			//
			pucSubnetMask[0] = 0xFF;
			pucSubnetMask[1] = 0xFF;
			pucSubnetMask[2] = 0xFF;
			pucSubnetMask[3] = 0x0;

			pucIP_Addr[0] = ascii_to_char(usBuffer[2], usBuffer[3]);
			pucIP_Addr[1] = ascii_to_char(usBuffer[4], usBuffer[5]);
			pucIP_Addr[2] = ascii_to_char(usBuffer[6], usBuffer[7]);
			pucIP_Addr[3] = ascii_to_char(usBuffer[8], usBuffer[9]);

			pucIP_DefaultGWAddr[0] = ascii_to_char(usBuffer[10], usBuffer[11]);
			pucIP_DefaultGWAddr[1] = ascii_to_char(usBuffer[12], usBuffer[13]);
			pucIP_DefaultGWAddr[2] = ascii_to_char(usBuffer[14], usBuffer[15]);
			pucIP_DefaultGWAddr[3] = ascii_to_char(usBuffer[16], usBuffer[17]);
			//
			// Currently no implementation of DHCP in hte demo
			//
			pucDNS[0] = 0;
			pucDNS[1] = 0;
			pucDNS[2] = 0;
			pucDNS[3] = 0;
			
			netapp_dhcp((unsigned long *)pucIP_Addr, (unsigned long *)pucSubnetMask, (unsigned long *)pucIP_DefaultGWAddr, (unsigned long *)pucDNS);
			
			break;
    	//
    	// Handle WLAN disconnect command
    	//
		case UART_COMMAND_CC3000_DISCONNECT:
			wlan_disconnect();
			break;

		
    	//
    	// Handle erase policy command
    	//
		case UART_COMMAND_CC3000_DEL_POLICY:
			wlan_ioctl_set_connection_policy(0, 0, 0);
			break;
			
		default:
			DispatcherUartSendPacket(pucUARTIllegalCommandString, sizeof(pucUARTIllegalCommandString));
			break;
			
	}
	//
	// Send a responce - the command handling has finished
	//
	DispatcherUartSendPacket(pucUARTCommandDoneString, sizeof(pucUARTCommandDoneString));
    
}

/**
  * @brief  This function turns the connection LED ON/OFF
  * @param  None
  * @retval None
  */
static void SmartConfigLedOn(uint8_t ulTrueFalse)
{
	if(ulTrueFalse)
	{
		halBoardLedOn(BIT0);	  
	}
	else
	{
		halBoardLedOff(BIT0);
	}
}

/**
  * @brief  This function handles asynchronous events that come from CC3000 device
  * 		and operates to indicate exchange of data
  * @param  The type of event we just received.
  * @retval None
  */
void CC3000_UsynchCallback(long lEventType, char * data, unsigned char length)
{
	if (lEventType == HCI_EVNT_WLAN_ASYNC_SIMPLE_CONFIG_DONE)
	{
		ulSmartConfigFinished = 1;
	}

	if (lEventType == HCI_EVNT_WLAN_UNSOL_CONNECT)
	{
		ulCC3000Connected = 1;
		/* Turn On LED */
		SmartConfigLedOn(TRUE);
	}

	if (lEventType == HCI_EVNT_WLAN_UNSOL_DISCONNECT)
	{		
		ulCC3000Connected = 0;
		/*  Turn Off LED */
		SmartConfigLedOn(FALSE);
	}
}

/**
  * @brief  Configure the boards SPI peripheral for communication with CC3000.
  * @param  None
  * @retval None
  */
static void DemoInitSpi(void)
{   

	
	/* Enable Interrupt on RF IRQ pin */
	halButtonsInit(BUTTON_IRQ);
    halButtonsInterruptEnable(BUTTON_IRQ);
    buttonsPressed = 0;
	WriteWlanPin(FALSE);
	/* Start operation of SPI module */
	halRfSpiInit();
}

/***************************************************************************//**
 * @brief  Handles TimerA1 interrupts to run the unsolicited event handler
 * @param  none
 * @return none
 ******************************************************************************/
#if 0
#pragma vector=TIMERB1_VECTOR
__interrupt void TIMERB1_ISR(void)
#else
// TimerA1 interrupt service routine
#pragma vector=TIMER1_A0_VECTOR
__interrupt void TIMERA1_ISR(void)
#endif

{
	/* Clear the timer interrupt flag and reload */
	TA1CCTL0 &= ~CCIFG;	
	/* Handle any un-solicited event if required - the function shall be triggered 
	   at least twice a second 
	*/ 
	hci_unsolicited_event_handler();
}

/**
  * @brief  Start the timer
  * @param  None
  * @retval None
  */
static void StartUnsolicitedEventTimer(void)
{

    TA1CTL = TASSEL_1 + TACLR;                      // ACLK, reset the timer divider	
	//Stop the Timer	
    TA1CTL &= ~MC_1;	
    // Timer_A will repeatedly run to 547, using 32kHz clock, and then generate
    // an interrupt that wakes the main loop (every 1/60 sec)
    TA1CCTL0 = CCIE;                                // CCR0 interrupt enabled
    TA1CCR0 = 16384;                                  // 16384/32768 = a period of 500ms
	//Stop the timer
	TA1CTL |= MC_0;
	 // Start timer
    TA1CTL |= MC_1; //Timer counts up to TAxCCR0	
}

/**
  * @brief  Initialize communication with CC3000 and prints out driver version.
  * @param  None
  * @retval None
  */
static int DemoInitDriver(void)
{
#if defined(__TI_COMPILER_VERSION__)
	unsigned char version_string[40];
#else
	__no_init unsigned char version_string[40];
#endif
	//Configure the Serial Interface
	DispatcherUARTConfigure();
	//	
    //Configuret the SPI Interface with CC3000
    //
    DemoInitSpi();
	//
	// WLAN On API Implementation
	//
	wlan_init( CC3000_UsynchCallback, (tFWPatches )sendWLFWPatch, (tDriverPatches)sendDriverPatch, (tBootLoaderPatches) sendBootLoaderPatch, ReadWlanInterruptPin, WlanInterruptEnable, WlanInterruptDisable, WriteWlanPin);

	//
	// Trigger a WLAN device
	//
	wlan_start(0);
	//
	// Mask out all non-required events from CC3000
	//
	wlan_set_event_mask(HCI_EVNT_WLAN_KEEPALIVE|HCI_EVNT_WLAN_UNSOL_INIT|HCI_EVNT_WLAN_UNSOL_DHCP|HCI_EVNT_WLAN_ASYNC_PING_REPORT);
	//
	// Generate teh event to CLI: send a version string
	//
	DispatcherUartSendPacket(pucUARTExampleAppString, sizeof(pucUARTExampleAppString));
        version_string[0]= 0x30 + PALTFORM_VERSION;
        version_string[1]= '.';
        version_string[2]= 0x30 + APPLICATION_VERSION_NUMBER;
        version_string[3]= '.';
        version_string[4] = 0x30 + SPI_VERSION_NUMBER;
        version_string[5]= '.';
        version_string[6]= 0x30 + DRIVER_VERSION_NUMBER;
        version_string[7]=  '\n';
        version_string[8]=  '\r';
        DispatcherUartSendPacket(version_string, 9);
		StartUnsolicitedEventTimer();		
    return(0);
}
/**********************************************************************//**
 * @brief  Checks for the board revision
 *
 * @param  none
 *
 * @return Whether or not the board revision matches the software
 * - 0 - The board revision does not match the software
 * - 1 - The board revision matches the software
 *************************************************************************/

unsigned char assert_board_version(void)
{

    P8DIR &= ~BIT7;                         // Set P8.7 input
    P8OUT |= BIT7;                          // Set pullup resistor
    P8REN |= BIT7;                          // Enable pull up resistors

    if (P8IN & BIT7)                        // Board rev != 0_03?
        return 0;                           // If so, indicate failure

    P8DIR |= BIT7;                          // Set P8.7 output
    P8OUT &= ~BIT7;                         // Set P8.7 = 0
    P8REN &= ~BIT7;                         // Disable pull up resistors

    return 1;
}

void main(void)
{
	uint32_t ulCounter =0;	
    // Stop WDT
    WDTCTL = WDTPW + WDTHOLD;
    /* Check for the version of the board. This code requires an
     * MSP-EXP430F5438 REV 0-03 (or later) to work. Trap code execution
     * if validation fails. */
    if (!assert_board_version()){
        while (1) ;
    }

    /* Check the silicon the code is running on. This code requires an
     * actual MSP430F5438A device to work. Trap code execution
     * if validation fails. */
    if (Get_Device_Type() != F5438A){
        while (1) ;
    }

    //Initialize clock and peripherals
    halBoardInit();
    LFXT_Start(XT1DRIVE_0);
    Init_FLL_Settle(16000, 488);
    // Set system clock to max (25MHz)	
    //Init_FLL_Settle(25000, 488);	
    SFRIE1 |= OFIE;
    // Globally enable interrupts
#if defined(__TI_COMPILER_VERSION__)
    __bis_SR_register(GIE);     // Enable interrupts
#else
    __enable_interrupt();
#endif
	/* Initialize Interface with CC3000 */
	DemoInitDriver();
	uart_have_cmd = 0;


	while(1) 
	{
		if ( uart_have_cmd )
		{	/* Delay here for a bit while all the chars are being read */
			for (ulCounter = 0; ulCounter < 100; ulCounter++)
				SysCtlDelay(10000);
			/* Process the data */
			DemoHandleUartCommand(g_ucRxBuffer);
			
			/*Reset the Rx Buffer pointer and counter*/ 	
			memset(g_ucRxBuffer,0,USART_IF_BUFFER);
			g_usRxBufferCounter =0;
			uart_have_cmd = 0;
		}
	}	
}
